/*
 * Decompiled with CFR 0.152.
 */
package net.minecraft.world.level.storage;

import com.mojang.datafixers.DataFixer;
import com.mojang.logging.LogUtils;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Optional;
import java.util.UUID;
import net.minecraft.SystemUtils;
import net.minecraft.nbt.GameProfileSerializer;
import net.minecraft.nbt.NBTCompressedStreamTools;
import net.minecraft.nbt.NBTReadLimiter;
import net.minecraft.nbt.NBTTagCompound;
import net.minecraft.server.level.EntityPlayer;
import net.minecraft.util.datafix.DataFixTypes;
import net.minecraft.world.entity.player.EntityHuman;
import net.minecraft.world.level.storage.Convertable;
import net.minecraft.world.level.storage.FileNameDateFormatter;
import net.minecraft.world.level.storage.SavedFile;
import org.bukkit.Bukkit;
import org.bukkit.craftbukkit.v1_21_R1.entity.CraftPlayer;
import org.slf4j.Logger;
import org.spigotmc.SpigotConfig;

public class WorldNBTStorage {
    private static final Logger LOGGER = LogUtils.getLogger();
    private final File playerDir;
    protected final DataFixer fixerUpper;
    private static final DateTimeFormatter FORMATTER = FileNameDateFormatter.create();

    public WorldNBTStorage(Convertable.ConversionSession convertable_conversionsession, DataFixer datafixer) {
        this.fixerUpper = datafixer;
        this.playerDir = convertable_conversionsession.getLevelPath(SavedFile.PLAYER_DATA_DIR).toFile();
        this.playerDir.mkdirs();
    }

    public void save(EntityHuman entityhuman) {
        if (SpigotConfig.disablePlayerDataSaving) {
            return;
        }
        try {
            NBTTagCompound nbttagcompound = entityhuman.saveWithoutId(new NBTTagCompound());
            Path path = this.playerDir.toPath();
            Path path1 = Files.createTempFile(path, entityhuman.getStringUUID() + "-", ".dat", new FileAttribute[0]);
            NBTCompressedStreamTools.writeCompressed(nbttagcompound, path1);
            Path path2 = path.resolve(entityhuman.getStringUUID() + ".dat");
            Path path3 = path.resolve(entityhuman.getStringUUID() + ".dat_old");
            SystemUtils.safeReplaceFile(path2, path1, path3);
        }
        catch (Exception exception) {
            LOGGER.warn("Failed to save player data for {}", (Object)entityhuman.getName().getString());
        }
    }

    private void backup(String name, String s1, String s2) {
        Path path = this.playerDir.toPath();
        Path path1 = path.resolve(s1 + s2);
        Path path2 = path.resolve(s1 + "_corrupted_" + LocalDateTime.now().format(FORMATTER) + s2);
        if (Files.isRegularFile(path1, new LinkOption[0])) {
            try {
                Files.copy(path1, path2, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
            }
            catch (Exception exception) {
                LOGGER.warn("Failed to copy the player.dat file for {}", (Object)name, (Object)exception);
            }
        }
    }

    private Optional<NBTTagCompound> load(String name, String s1, String s2) {
        File file = this.playerDir;
        File file1 = new File(file, s1 + s2);
        boolean usingWrongFile = false;
        if (!file1.exists() && (file1 = new File(file, UUID.nameUUIDFromBytes(("OfflinePlayer:" + name).getBytes(StandardCharsets.UTF_8)).toString() + s2)).exists()) {
            usingWrongFile = true;
            Bukkit.getServer().getLogger().warning("Using offline mode UUID file for player " + name + " as it is the only copy we can find.");
        }
        if (file1.exists() && file1.isFile()) {
            try {
                Optional<NBTTagCompound> optional = Optional.of(NBTCompressedStreamTools.readCompressed(file1.toPath(), NBTReadLimiter.unlimitedHeap()));
                if (usingWrongFile) {
                    file1.renameTo(new File(file1.getPath() + ".offline-read"));
                }
                return optional;
            }
            catch (Exception exception) {
                LOGGER.warn("Failed to load player data for {}", (Object)name);
            }
        }
        return Optional.empty();
    }

    public Optional<NBTTagCompound> load(EntityHuman entityhuman) {
        return this.load(entityhuman.getName().getString(), entityhuman.getStringUUID()).map(nbttagcompound -> {
            if (entityhuman instanceof EntityPlayer) {
                CraftPlayer player = (CraftPlayer)entityhuman.getBukkitEntity();
                long modified = new File(this.playerDir, entityhuman.getStringUUID() + ".dat").lastModified();
                if (modified < player.getFirstPlayed()) {
                    player.setFirstPlayed(modified);
                }
            }
            entityhuman.load((NBTTagCompound)nbttagcompound);
            return nbttagcompound;
        });
    }

    public Optional<NBTTagCompound> load(String name, String uuid) {
        Optional<NBTTagCompound> optional = this.load(name, uuid, ".dat");
        if (optional.isEmpty()) {
            this.backup(name, uuid, ".dat");
        }
        return optional.or(() -> this.load(name, uuid, ".dat_old")).map(nbttagcompound -> {
            int i2 = GameProfileSerializer.getDataVersion(nbttagcompound, -1);
            nbttagcompound = DataFixTypes.PLAYER.updateToCurrentVersion(this.fixerUpper, (NBTTagCompound)nbttagcompound, i2);
            return nbttagcompound;
        });
    }

    public File getPlayerDir() {
        return this.playerDir;
    }
}

