package org.bukkit;

import static org.bukkit.support.MatcherAssert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.jupiter.api.Assertions.*;
import com.google.common.collect.Maps;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.StreamSupport;
import net.minecraft.core.registries.BuiltInRegistries;
import net.minecraft.resources.MinecraftKey;
import net.minecraft.world.item.Item;
import org.bukkit.craftbukkit.util.CraftMagicNumbers;
import org.bukkit.support.LegacyHelper;
import org.bukkit.support.environment.AllFeatures;
import org.junit.jupiter.api.Test;

@AllFeatures
public class MaterialTest {

    @Test
    public void verifyMapping() {
        Map<MinecraftKey, Material> materials = Maps.newHashMap();
        for (Material material : Material.values()) {
            if (LegacyHelper.getInvalidatedMaterials().contains(material)) {
                continue;
            }

            materials.put(CraftMagicNumbers.key(material), material);
        }

        Iterator<Item> items = BuiltInRegistries.ITEM.iterator();

        while (items.hasNext()) {
            Item item = items.next();
            if (item == null) continue;

            MinecraftKey id = BuiltInRegistries.ITEM.getKey(item);
            String name = item.getDescriptionId();

            Material material = materials.remove(id);

            assertThat(material, is(not(nullValue())), "Missing " + name + "(" + id + ")");
            assertNotNull(CraftMagicNumbers.getMaterial(item), "No item mapping for " + name);
        }

        assertThat(materials, is(Collections.EMPTY_MAP));
    }

    @Test
    public void verifyMaterialOrder() {
        List<Material> expectedOrder = new ArrayList<>(Material.values().length);

        // Start with items in the same order as BuiltInRegistries.ITEM
        StreamSupport.stream(BuiltInRegistries.ITEM.spliterator(), false)
                .map(CraftMagicNumbers::getMaterial)
                .forEach(expectedOrder::add);

        // Then non-item blocks in the same order as BuiltInRegistries.BLOCK
        StreamSupport.stream(BuiltInRegistries.BLOCK.spliterator(), false)
                .map(CraftMagicNumbers::getMaterial)
                .filter(block -> !block.isItem())
                .forEach(expectedOrder::add);

        // Then legacy materials in order of ID
        Arrays.stream(Material.values())
                .filter(Material::isLegacy)
                .sorted(Comparator.comparingInt(Material::getId))
                .forEach(expectedOrder::add);

        assertArrayEquals(expectedOrder.toArray(), Material.values());
    }
}
